#!/usr/bin/perl
#
# Copyright 2007-2023 Michel Messerschmidt
# License: GPL-3+
# 
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.

use strict;
use warnings;

use File::Copy;
use File::Temp qw/ tempfile /;
use Cwd qw/ abs_path /;
use Data::Dumper;

$::PROG="noCA.pl";
$::VERSION = "1.1";

sub parse_ca_config($);
sub parse_alg_config($);
sub parse_cert_config($);
sub parse_req_config($);
sub ossl_get_digests();
sub ossl_get_curves();
sub cmd_init();
sub cmd_edit($);
sub cmd_newca();
sub cmd_newsubca();
sub cmd_req();
sub cmd_sign($);
sub cmd_revoke($);
sub cmd_selfsign($$);
sub cmd_createp12(@);
sub cmd_gencrl();
sub cmd_exportca();
sub cmd_cleanup();

my $verbose = 0;
# parse only verbosity parameters before loading config files
foreach my $i (@::ARGV) {
	if ($i =~ /^(-d|--debug)$/) {
		$verbose = 5;
	} elsif ($i =~ /^(-v|--verbose)$/) {
		$verbose++;
        }
}

### default configuration ###

# directories (note that cwd returns an absolute path)
my $pkidir = &Cwd::cwd();
$pkidir .= '/' unless (substr($pkidir, -1, 1) eq '/');
# be portable and try to work out of the current directory as much as possible
my $cnf_dir = $pkidir;

# openssl binary
my $openssl="openssl";
if(defined $ENV{OPENSSL}) {
	$openssl = $ENV{OPENSSL};
} else {
	$ENV{OPENSSL} = $openssl;
}
if(defined $ENV{OPENSSLCONF}) {
	$cnf_dir = $ENV{OPENSSLCONF};
} 
$cnf_dir .= '/' unless (substr($cnf_dir, -1, 1) eq '/');

# file names of keys and certificates
my $root_ca_key="private/root_ca_key.pem";
my $root_ca_cert="root_ca_cert.pem";
my $file_ca_certs_export_pkcs7="ca_certs.p7c";

# file names of openssl configuration files
my $file_conf_ca = "oca.cnf";
my $file_conf_alg = "oalg.cnf";
my $file_conf_ext = "oext.cnf";
my $file_conf_ext_ca = "oext_ca.cnf";
my $file_conf_req = "oreq.cnf";

# format of private key files
# use the "new" PKCS#8 format by default.
# set to 0 to use old SSLEAY format (OpenSSL 0.9.x)
my $keyformat = 1;

# symmetric encryption algorithm for private key files
# old default was -des3
my $keyenc = "aes-128-cbc";

# holds all CA configuration settings
my %ca_conf = ();
# list of environment variables that need to be set for CA certificate signing requests
my @env_ca_req = ();
# list of available CAs
my @ca_avail = ( "root_ca", "sub1_ca", "sub2_ca" );
# Name of the root CA must match openssl config, hardcoded on purpose to 
# prevent CA hierarchy tree confusion due to configuration errors
my $root_ca = "root_ca";
# CA that is selected to perform the requested action
my $ca = $root_ca;

# certificate lifetimes (in days)
#  The root CA lifetime cannot be specified in the ca config file
#  It uses either the hardcoded value of 20 years here or can be modified with the --days option
my $lifetime_root_ca = "7305";
#  Note: All other certificate lifetimes are specified in the ca config file

# supported hash algorithms (by openssl)
my $refossld = ossl_get_digests();
my @hash_avail = @{$refossld};
undef $refossld;

# asymmetric public key crypto algorithms for digital signatures and certificates, supported by OpenSSL version 3.5
my @dsig_avail = ( "ML-DSA-65", "ML-DSA-87", "ML-DSA-44", "EC", "RSA-PSS", "ED25519", "ED448", "RSA" );
my $dsig = $dsig_avail[0];

my %alg_avail = ();
# explicitly create empty option hash for each algorithm because not all algorithms have options
#foreach (@{$alg_avail{"default"}{"publickey"}}) {
foreach (@dsig_avail) {
	%{$alg_avail{$_}} = ();
}
# openssl keygen options are hardcoded because not that simple to retrieve from openssl
# the first list value is always the recommended default
# openssl RSA keygen options
@{$alg_avail{"RSA"}{rsa_keygen_bits}} = ( 4096, 3072, 6144, 8192, 2048, 1024, 512, "another value");
@{$alg_avail{"RSA"}{rsa_keygen_primes}} = ( 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 );
@{$alg_avail{"RSA"}{rsa_keygen_pubexp}} = ( 65537, 3 );
# openssl RSA-PSS keygen options
@{$alg_avail{"RSA-PSS"}{rsa_keygen_bits}} = ( 4096, 3072, 6144, 8192, 2048, 1024, 512, "another value" );
@{$alg_avail{"RSA-PSS"}{rsa_keygen_primes}} = ( 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 );
@{$alg_avail{"RSA-PSS"}{rsa_keygen_pubexp}} = ( 65537, 3 );
# These additional RSA-PSS parameters are usage restrictions and not used by default
@{$alg_avail{"RSA-PSS"}{rsa_pss_keygen_md}} = ("not set", @hash_avail);
@{$alg_avail{"RSA-PSS"}{rsa_pss_keygen_mgf1_md}} = ("not set", @hash_avail);
@{$alg_avail{"RSA-PSS"}{rsa_pss_keygen_saltlen}} = ( "not set", 32, 48, 64, 20, 28, 16, 0 );
# openssl EC keygen options
# retrieve supported curves from openssl
my $refosslc = ossl_get_curves();
@{$alg_avail{"EC"}{ec_paramgen_curve}} = @{$refosslc};
undef $refosslc;
# Note: explicit curve parameter specification is supported by openssl but not by this script
@{$alg_avail{"EC"}{ec_param_enc}} = ( "named_curve" );
# openssl X25519 keygen has no options
# openssl X448 keygen has no options
# openssl ED25519 keygen has no options
# openssl ED448 keygen has no options
# ML-DSA hexseed option is not made available because usage is explicitly discouraged
#@{$alg_avail{"ML-DSA-65"}{hexseed}} = ( "" );
#@{$alg_avail{"ML-DSA-87"}{hexseed}} = ( "" );
#@{$alg_avail{"ML-DSA-44"}{hexseed}} = ( "" );

my %alg = ();
$alg{"default"}{"publickey"} = $dsig;
%{$alg{$dsig}} = ();
foreach my $k (keys %{$alg_avail{$dsig}}) {
	$alg{$dsig}{$k} = $alg_avail{$dsig}{$k}[0];
}

# supported certificate revocation reasons
# Note that removeFromCRL is not include here because delta CRLs are not supported by this script
# Note that RFC 5280 also specifies privilegeWithdrawn and aACompromise but these are not supported in openssl version 3.1.3 or older
my @revoke_reasons = ( "NONE", "unspecified", "keyCompromise", "cACompromise", "affiliationChanged", "superseded", "cessationOfOperation", "certificateHold" );

### end of default configuration ###

# instance configuration
my %opt = ();
# update default configuration with config file settings
if (-f $pkidir . $file_conf_ca) {
	my ($calistref, $caconfref, $envreqref);
	# update default CA, CA list and CA config
	($ca, $calistref, $caconfref, $envreqref) = parse_ca_config($pkidir . $file_conf_ca);
	@ca_avail = @{$calistref};
	%ca_conf = %{$caconfref};
	@env_ca_req = @{$envreqref};
	# update root CA filenames and lifetimes from config file
	$root_ca_key = $ca_conf{$root_ca}{"private_key"};
	$root_ca_cert = $ca_conf{$root_ca}{"certificate"};
} else {
	print "WARNING: Configuration file not found: ", $pkidir, $file_conf_ca, "\n";
	print "WARNING: Using default configuration\n";
}
if (-f $pkidir . $file_conf_alg) {
	my $algref;
	$algref = parse_alg_config($pkidir . $file_conf_alg);
	%alg = %{$algref};
	$dsig = $alg{"default"}{"publickey"};
} else {
	print "WARNING: Configuration file not found: ", $pkidir, $file_conf_alg, "\n";
	print "WARNING: Using default configuration\n";
}

### end of instance configuration ###

# runtime configuration
my $ret = 0;
while (defined($::ARGV[0]) and $::ARGV[0] =~ /^-/) {
	if ($::ARGV[0] =~ /^(-\?|-h|-?-help)$/ ) {
		main::HELP_MESSAGE();
		exit 0;
	} elsif ($::ARGV[0] =~ /^(-V|-?-version)$/) {
	    	print("$::PROG $::VERSION\n");
		exit 0;
	} elsif ($::ARGV[0] =~ /^(-v|--verbose|-d|--debug)$/) {
		# debug otions were handled earlier
		shift @::ARGV;
	} elsif ($::ARGV[0] =~ /^--hash(=.*)?$/ ) {
		shift @::ARGV;
		if  (defined $1) {
			$opt{'md'} = substr($1, 1);
		} elsif (defined $::ARGV[0]) {
			$opt{'md'} = $::ARGV[0];
			shift @::ARGV;
		} else {
			print "ERROR: --hash option requires an argument\n";
			exit 1;
		}
	} elsif ($::ARGV[0] =~ /^--size(=.*)?$/ ) {
		shift @::ARGV;
		if  (defined $1) {
			$opt{'keysize'} = substr($1, 1);
		} elsif (defined $::ARGV[0]) {
			$opt{'keysize'} = $::ARGV[0];
			shift @::ARGV;
		} else {
			print "ERROR: --size option requires an argument\n";
			exit 1;
		}
	} elsif ($::ARGV[0] =~ /^--curve(=.*)?$/ ) {
		shift @::ARGV;
		if  (defined $1) {
			$opt{'curve'} = substr($1, 1);
		} elsif (defined $::ARGV[0]) {
			$opt{'curve'} = $::ARGV[0];
			shift @::ARGV;
		} else {
			print "ERROR: --curve option requires an argument\n";
			exit 1;
		}
		# set new curve globally 
		$alg{"EC"}{'ec_paramgen_curve'} = $opt{'curve'};
	} elsif ($::ARGV[0] =~ /^--start(=.*)?$/ ) {
		shift @::ARGV;
		if  (defined $1) {
			$opt{'start'} = substr($1, 1);
		} elsif (defined $::ARGV[0]) {
			$opt{'start'} = $::ARGV[0];
			shift @::ARGV;
		} else {
			print "ERROR: --start option requires an argument\n";
			exit 1;
		}
		if ($opt{'start'} !~ /^[0-9]?[0-9]?[0-9]{12}Z$/) {
			print "ERROR: --start format must be YYMMDDHHMMSSZ or YYYYMMDDHHMMSSZ\n";
			exit 1;
		}
	} elsif ($::ARGV[0] =~ /^--end(=.*)?$/ ) {
		shift @::ARGV;
		if  (defined $1) {
			$opt{'end'} = substr($1, 1);
		} elsif (defined $::ARGV[0]) {
			$opt{'end'} = $::ARGV[0];
			shift @::ARGV;
		} else {
			print "ERROR: --end option requires an argument\n";
			exit 1;
		}
		if ($opt{'end'} !~ /^[0-9]?[0-9]?[0-9]{12}Z$/) {
			print "ERROR: --end format must be YYMMDDHHMMSSZ or YYYYMMDDHHMMSSZ\n";
			exit 1;
		}
	} elsif ($::ARGV[0] =~ /^--days(=.*)?$/ ) {
		shift @::ARGV;
		if  (defined $1) {
			$opt{'days'} = int(substr($1, 1));
		} elsif (defined $::ARGV[0]) {
			$opt{'days'} = int($::ARGV[0]);
			shift @::ARGV;
		} else {
			print "ERROR: --days option requires an argument\n";
			exit 1;
		}
	} elsif ($::ARGV[0] =~ /^--keyenc(=.*)?$/ ) {
		shift @::ARGV;
		if  (defined $1) {
			$keyenc = substr($1, 1);
			$opt{'keyenc'} = substr($1, 1);
		} elsif (defined $::ARGV[0]) {
			$keyenc = $::ARGV[0];
			$opt{'keyenc'} = $::ARGV[0];
			shift @::ARGV;
		} else {
			print "ERROR: --keyenc option requires an argument\n";
			exit 1;
		}
	} elsif ($::ARGV[0] =~ /^--oldkey$/) {
		shift @::ARGV;
		$keyformat = 0;
	} else {
		print "ERROR: unknown option.\n";
		main::HELP_MESSAGE();
		exit 1;
	}
}
### end of runtime configuration ###

if ($verbose > 4) {
	print "DEBUG: configuration:\n";
	print "  root_ca = ",$root_ca,"\n";
	print "  root_ca_certificate = ",$root_ca_cert,"\n";
	print "  root_ca_key = ",$root_ca_key,"\n";
	print "  dsig = ",$dsig,"\n";
	print "  keyformat=",$keyformat,"\n";
	print "  keyenc=",$keyenc,"\n";
}
if ($verbose > 5) {
	print Data::Dumper->Dump([\@ca_avail, \@hash_avail, \@dsig_avail, \%alg_avail, \%ca_conf, \@env_ca_req, \%alg, \%opt], [qw(*ca_avail *hash_avail *dsig_avail *alg_avail *ca_conf *env_ca_req *alg *opt)]),"\n";
	exit 0;
}

# some simple configuration checks 
if (($keyformat == 0) and $dsig ne "RSA") {
	print "ERROR: Old keyformat is only supported for RSA keys.\n";
	exit 1;
}
if (exists $opt{'curve'}) {
	my $known = 0;
	foreach (@{$alg_avail{"EC"}{ec_paramgen_curve}}) {
		if ($_ eq $opt{'curve'}) {
			$known = 1;
		}
	}
	if ($known == 0) {
		print "ERROR: Specified EC named curve not supported by the OpenSSL version in use\n";
		exit 1;
	}
}
if (exists $opt{'md'}) {
	my $known = 0;
	foreach (@hash_avail) {
		if ($_ eq $opt{'md'}) {
			$known = 1;
		}
	}
	if ($known == 0) {
		print "ERROR: Specified hash algorithm not supported by the OpenSSL version in use\n";
		exit 1;
	}
}
if ((exists $opt{'end'}) and (exists $opt{'days'})) {
	print "ERROR: --end and --days options cannot be used together\n";
	exit 1;
}
### end of configuration checks ###

# let's go
if (defined($::ARGV[0])) {
	if ($::ARGV[0] =~ /^help$/) {
		main::HELP_MESSAGE();
		exit 0;
	} elsif ($::ARGV[0] =~ /^init$/) {
		$ret = cmd_init();
	} elsif ($::ARGV[0] =~ /^edit$/) {
		$ret = cmd_edit("");
	} elsif ($::ARGV[0] =~ /^newca$/) {
		$ret = cmd_newca();
	} elsif ($::ARGV[0] =~ /^newsubca$/) {
		$ret = cmd_newsubca();
	} elsif ($::ARGV[0] =~ /^req$/) {
		$ret = cmd_req();
	} elsif ($::ARGV[0] =~ /^sign$/) {
		shift @::ARGV;
		if (not defined $::ARGV[0]) {
		    print "ERROR: missing input file\n";
		    main::HELP_MESSAGE();
		    exit 2;
		}
		$ret = cmd_sign($::ARGV[0]);
	} elsif ($::ARGV[0] =~ /^selfsign$/) {
		shift @::ARGV;
		if (not defined $::ARGV[0] or not defined $::ARGV[1] or $#::ARGV > 1) {
		    print "ERROR: wrong number of input files\n";
		    main::HELP_MESSAGE();
		    exit 2;
		}
		$ret = cmd_selfsign($::ARGV[0], $::ARGV[1]);
	} elsif ($::ARGV[0] =~ /^revoke$/) {
		shift @::ARGV;
		if (not defined $::ARGV[0]) {
		    print "ERROR: missing input file\n";
		    main::HELP_MESSAGE();
		    exit 2;
		}
		$ret = cmd_revoke($::ARGV[0]);
	} elsif ($::ARGV[0] =~ /^p12$/) {
		shift @::ARGV;
		if (not defined $::ARGV[0]) {
		    print "ERROR: missing output file\n";
		    main::HELP_MESSAGE();
		    exit 2;
		} elsif (not defined $::ARGV[1]) {
		    print "ERROR: missing input files\n";
		    main::HELP_MESSAGE();
		    exit 2;
		}
		$ret = cmd_createp12(@::ARGV);
	} elsif ($::ARGV[0] =~ /^gencrl$/) {
		$ret = cmd_gencrl();
	} elsif ($::ARGV[0] =~ /^exportca$/) {
		$ret = cmd_exportca();
	} elsif ($::ARGV[0] =~ /^cleanup$/) {
		$ret = cmd_cleanup();
	} else {
		print STDERR "ERROR: Unknown command\n";
		main::HELP_MESSAGE();
		exit 1;
	}
} else {
	print "ERROR: missing command.\n";
	main::HELP_MESSAGE();
	exit 1;
}

exit $ret;


sub cmd_init() {
	if (-f $ca_conf{$root_ca}{'dir'} . $ca_conf{$root_ca}{'database'}) {
		print "ERROR: PKI already initialized\n";
		return 1;
	}
	print "Initializing PKI...\n";
	my $perm_dir = 0777;
	my $perm_dir_sec = 0700;
	# create main PKI directories
	mkdir $pkidir, $perm_dir unless (-d $pkidir);
	mkdir $pkidir . 'export', $perm_dir_sec unless (-d $pkidir . 'export');

	if ($cnf_dir ne $pkidir) {
		foreach my $cf ($file_conf_ca, $file_conf_alg, $file_conf_req, $file_conf_ext, $file_conf_ext_ca) {
			copy($cnf_dir . $cf, $pkidir . $cf) or 
			    die "ERROR: failed to copy template file ". $cf ."from ". $cnf_dir .": ". $!;
		}
		foreach my $c (@ca_avail) {
			# cut '_ca' from CA config section name to get actual name
			my $ca_name = substr($c, 0, -3); 
			if (-f $cnf_dir . $file_conf_ext . '_' . $ca_name) {
				copy($cnf_dir . $file_conf_ext . '_' . $ca_name, 
				     $pkidir . $file_conf_ext . '_' . $ca_name) or 
			    	     print "WARNING: failed to copy template file ". 
				           $file_conf_ext . '_' . $ca_name ."from ". $cnf_dir .": ". $!;
			}
		}
	} else {
		foreach my $cf ($file_conf_ca, $file_conf_alg, $file_conf_req, $file_conf_ext, $file_conf_ext_ca) {
			unless (-r $pkidir . $cf) {
				die "ERROR: required config file ". $cf .
			    	    " not found. Please copy it manually to ". $pkidir ."\n"; 
			    }
		    }
	}

	# create the configured directories and files for each CA
	foreach my $c (@ca_avail) {
		# CA config 'dir' entry has been converted into absolute path during config read
		my $cad = $ca_conf{$c}{'dir'};
		mkdir $cad, $perm_dir unless (-d $cad);
		# some directory names are hardcoded
		mkdir $cad . 'requests', $perm_dir unless (-d $cad . 'requests');
		mkdir $cad . 'private', $perm_dir_sec unless (-d $cad . 'private');
		# some directory names are taken from openssl configuration file
		foreach my $d ('certs', 'crl_dir', 'new_certs_dir') {
			mkdir $cad . $ca_conf{$c}{$d}, $perm_dir unless (-d $cad . $ca_conf{$c}{$d});
		}
		# filenames taken from openssl configuration file
		unless (-f $cad . $ca_conf{$c}{'serial'}) {
			open OUT, '>', $cad . $ca_conf{$c}{'serial'};
			# Set the initial serial number to a 20 byte pseudo-random value
			# because CAB/F requires at least 20 bytes entropy in the serial number
			# This is not a cryptographically secure random number, but does not need to be here
			my $s = "";
			for (my $i = 0; $i < 20; $i++) {
				# Ensure zero padding as openssl expects an even number of uppercase hex digits
				$s .= sprintf("%02X", int(rand(0x100)));
			}
			print OUT $s,"\n";
			close OUT;
		}
		unless (-f $cad . $ca_conf{$c}{'crlnumber'}) {
			open OUT, '>', $cad . $ca_conf{$c}{'crlnumber'};
			print OUT "01\n";
			close OUT;
		}
		# openssl ca database file
		unless (-f $cad . $ca_conf{$c}{'database'}) {
			open OUT, '>', $cad . $ca_conf{$c}{'database'};
			close OUT;
		}
		# openssl ca uses an additional file to keep track of CA attributes
		unless (-f $cad . $ca_conf{$c}{'database'} . '.attr') {
			open OUT, '>', $cad . $ca_conf{$c}{'database'} . '.attr';
			close OUT;
		}
	}
	print "...PKI initialization done.\n";
	# customize default configuration
	my $r = cmd_edit($root_ca);
	return $r unless ($r == 0);
	
	print "Create root CA key pair and certificate?\n";
	print "Enter y or just ENTER to proceed, any other input to abort: ";
	chomp(my $answer_newca=<STDIN>);
	print "\n";
	if ($answer_newca ne "" and $answer_newca ne "y") {
		print "Use the newca command to create the root CA certificate\n";
		return $r;
	}
	$r = cmd_newca();
	return $r unless ($r == 0);

	print "Create a sub CA key pair and certificate?";
	print "Enter y or just ENTER to proceed, any other input to abort: ";
	chomp(my $answer_subca=<STDIN>);
	print "\n";
	if ($answer_subca ne "" and $answer_subca ne "y") {
		print "...Root CA initialization completed.\n";
		print "Use the newsubca command to create a sub CA.\n";
		return $r;
	}
	$r = cmd_newsubca();
	return $r;
}


# customize default configuration
sub cmd_edit($) {
	print "Customizing CA...\n";
	my $r = 0;
	if ($_[0] eq $root_ca) {
		$ca = $root_ca;
	} else {
		print "Which CA should be customized (ENTER = 1): \n";
		for (my $i = 0; $i <= $#ca_avail; $i++) {
			printf "  %2d - %s\n", $i+1, $ca_avail[$i];
		}
		chomp(my $answer_ca=<STDIN>);
		$answer_ca = 1 unless ($answer_ca ne "");
		# convert input back to array indices
		$answer_ca = int($answer_ca) - 1;
		if ($answer_ca < 0 or $answer_ca > $#ca_avail) {
			print "ERROR: invalid choice.\n";
			return 2;
		} else {
			$ca = $ca_avail[$answer_ca];
		}
	}

	print "Which hash algorithm should be used (ENTER = 1): \n";
	for (my $i = 0; $i <= $#hash_avail; $i++) {
		printf "  %2d - %s\n", $i+1, $hash_avail[$i];
	}
	chomp(my $answer_hash=<STDIN>);
	$answer_hash = 1 unless ($answer_hash ne "");
	# convert input back to array indices
	$answer_hash = int($answer_hash) - 1;
	if ($answer_hash < 0 or $answer_hash > $#hash_avail) {
		print "ERROR: invalid choice. Default '". $ca_conf{"$ca"}{"default_md"} ."' not changed.\n";
	} else {
		$ca_conf{"$ca"}{"default_md"} = $hash_avail[$answer_hash];
        }

	print "Which public key algorithm should be used (ENTER = 1): \n";
	print "Info: Algorithm is set globally and not per CA.\n";
	for (my $i = 0; $i <= $#dsig_avail; $i++) {
		printf "  %2d - %s\n", $i+1, $dsig_avail[$i];
	}
	chomp(my $answer_dsig=<STDIN>);
	$answer_dsig = 1 unless ($answer_dsig ne "");
	# convert input back to array indices
	$answer_dsig = int($answer_dsig) - 1;
	if ($answer_dsig < 0 or $answer_dsig > $#dsig_avail) {
		print "ERROR: invalid choice. Default '". $dsig ."' not changed.\n";
	} else {
		$dsig = $dsig_avail[$answer_dsig];
		$alg{"default"}{"publickey"} = $dsig_avail[$answer_dsig];
	}

	my @opts = keys %{$alg_avail{$dsig}};
	if ($#opts == -1) {
		print "\nInfo: Algorithm has no configurable options.\n";
	} else {
		print "\nConfiguring algorithm options...\n";
		print "The first entry for each option is the recommended default value.\n";
		print "Only select other values if you know exactly what you are doing!\n";
		print "There are no further checks to prevent invalid combinations!\n\n";
		foreach my $opt (@opts) {
			print "Which value should be used for ",$opt,"?\n";
			my @choices = @{$alg_avail{$dsig}{$opt}};
			for (my $i = 0; $i <= $#choices; $i++) {
				printf "  %2d - %s\n", $i+1, $choices[$i];
			}
			chomp(my $answer=<STDIN>);
			$answer = 1 unless ($answer ne "");
			# convert input back to array indices
			$answer = int($answer) - 1;
			if ($answer < 0 or $answer > $#choices) {
				print "ERROR: invalid choice. Default '". $choices[0] ."' not changed.\n";
			} else {
				$alg{$dsig}{$opt} = $choices[$answer];
			}
			if ($alg{$dsig}{$opt} eq "another value" ) {
				print "Enter the value for ",$opt," (no error checking): ";
				chomp(my $av=<STDIN>);
				$alg{$dsig}{$opt} = $av;
			}
			# remove key and value from list if the option should not be used
			if ($alg{$dsig}{$opt} eq "not set" ) {
				delete $alg{$dsig}{$opt};
			}
			print "\n";
		}
	}

	# update CA config file with new settings
	print "DEBUG: modifying ca config file ",$file_conf_ca,"\n" if ($verbose > 4);
	my @lines = ();
	my $edit = 0;
	open CACONFFILE, "<", $pkidir . $file_conf_ca or die "ERROR: failed to read openssl configuration file ". $pkidir .  $file_conf_ca ."\n";
	while (my $line=<CACONFFILE>) {
		print "DEBUG: read ca config line: ",$line if ($verbose > 5);
		# only update entries in the section of the selected CA
		if($line =~ /^\s*\[\s*$ca\s*\]\s*$/) {
			$edit = 1;
		}
		if (($edit == 1) and ($line =~ /^\s*default_md\s*=/)) {
			$line =~ s/default_md\s*=\s*(\S+)/default_md \t= $ca_conf{"$ca"}{"default_md"}/i;
			print "DEBUG: found match, is now: ",$line if ($verbose > 4);
			$edit = 0;
		}
		push @lines, $line;
	}
	close CACONFFILE;
	copy($pkidir . $file_conf_ca, $pkidir . $file_conf_ca . ".bak") or warn "WARNING: failed to backup config file ". $file_conf_ca .": ". $!;
	open CACONFFILE, ">", $pkidir . $file_conf_ca or die "ERROR: failed to write config file ". $pkidir . $file_conf_ca ."\n";
	foreach my $l (@lines) {
		print(CACONFFILE $l);
    	}
	close CACONFFILE;

	copy($pkidir . $file_conf_alg, $pkidir . $file_conf_alg . ".bak") or warn "WARNING: failed to backup config file ". $file_conf_alg .": ". $!;
	open ALGCONFFILE, ">", $pkidir . $file_conf_alg or die "ERROR: failed to write config file ". $pkidir . $file_conf_alg ."\n";
	print ALGCONFFILE "# noCA algorithm configuration file\n";
	print ALGCONFFILE "\n[ default ]\n";
	foreach my $k (keys %{$alg{"default"}}) {
		print ALGCONFFILE $k, " = ", $alg{"default"}{$k}, "\n";
	}
	foreach my $ka (@dsig_avail) {
		print ALGCONFFILE "\n[ ", $ka, " ]\n";
		foreach my $ko (keys %{$alg{$ka}}) {
			print ALGCONFFILE $ko, " = ", $alg{$ka}{$ko}, "\n";
		}
	}
	close ALGCONFFILE;
	print "...",$ca," configuration completed.\n";
	return $r;
}


sub cmd_newca() {
	if (not -d $pkidir . 'private' or
	    not -r $pkidir . $ca_conf{$root_ca}{'database'} or
	    not -r $pkidir . $file_conf_ca) { 
		print "ERROR: PKI not initialized. Run '", $::PROG, " init' first.\n";
		return 1;
	}
	if ( -f $pkidir . $root_ca_cert ) {
		print "ERROR: Root CA certificate exists. CA creation aborted.\n";
		print "It is recommended not to replace the root certificate ".
		      "but create another PKI instance for a new root CA.\n";
		return 1;
	}
	# determine the hash algorithm to use in the CSR, CLI has priority over config file 
	my $md = $ca_conf{$root_ca}{'default_md'};
	if (exists $opt{'md'}) {
		$md = $opt{'md'};
	}
	# determine serial number
	print "Do you want to specify the serial number for the root certificate?\n".
	      "Press ENTER to let openssl generate a large random number (recommended). \n".
	      "Otherwise enter a decimal or hexadecimal number. \n".
	      "Note that best practice are 20 bytes length for the serial number (not possible with decimal entries):\n";
	chomp(my $serno=<STDIN>);
	if ($serno =~ /^-?[0-9]+$/) {
		# convert input to integer
		$serno = int($serno);
	} elsif ($serno =~ /^(0x)?[0-9A-Fa-f]+$/) {
		if ($1 ne "0x") {
			$serno = "0x" . $serno;
		}
	} elsif ($serno ne "") {
	    print "ERROR: invalid input. No certificate created.\n";
	    exit 2;
	}
	my $certlife = "-days ". $lifetime_root_ca ." ";
	if (exists $opt{'days'}) {
		$certlife = "-days ". $opt{'days'} ." ";
	}
	if (exists $opt{'start'}) {
		print "WARNING: --start option not supported for root CA certificate.\n";
	}
	if (exists $opt{'end'}) {
		print "WARNING: --end option not supported for root CA certificate.\n";
	}

	print "Generating new Root CA certificate with ",$dsig," key and ",$md," hash algorithm...\n";
	my $cmd = "umask 0077; ". $openssl ." req -config ". $file_conf_ca ." ".
		"-new -x509 ".
		"-newkey ". $dsig ." ";
	foreach my $o (keys %{$alg{$dsig}}) {
		$cmd .= "-pkeyopt ". $o .":". $alg{$dsig}{$o} . " ";
	}
	if ($serno ne "") {
		$cmd .= "-set_serial ". $serno . " ";
	}
	$cmd .= "-" . $md ." ".
		$certlife ." ".
		"-keyout ". $pkidir . $root_ca_key ." ".
		"-out ". $pkidir . $root_ca_cert;
	print $cmd."\n" if ($verbose > 0);
	system ($cmd);
	print "...done.\n";
	return $? >> 8;
}


sub cmd_newsubca() {
	if ( ! -r $pkidir . $root_ca_key) {
		print "ERROR: Root CA private key not found. Run '", $::PROG, " newca' first.\n";
		return 1;
	}
	print "Which sub CA should be created:\n";
	my $ri = 0;
	for (my $i = 0; $i <= $#ca_avail; $i++) {
		if ($ca_avail[$i] ne $root_ca) {
			printf "  %2d - %s\n", $i+1, $ca_avail[$i];
		} else {
			$ri = $i;
		}
	}
	chomp(my $answer_ca=<STDIN>);
	# convert input back to array indices
	$answer_ca = int($answer_ca) - 1;
	if (($answer_ca < 0) or ($answer_ca > $#ca_avail) or ($answer_ca == $ri)) {
		print "ERROR: invalid choice.\n";
		return 2;
	} else {
		$ca = $ca_avail[$answer_ca];
	}
	if ( -e $ca_conf{$ca}{"dir"} . $ca_conf{$ca}{"certificate"} ) {
		print "WARNING: Sub CA certificate exists. Replace it (y/n)?\n";
		chomp(my $answer=<STDIN>);
		if ($answer ne "y") {
			print "Sub CA certificate generation aborted.\n";
			exit 0;
		}
	}
	my $conf = $pkidir . $file_conf_ext_ca;
	my $profileref = parse_cert_config($conf);
	my %profiles = %{$profileref};
	my @profno = sort keys %profiles;
	print "Generating new sub CA certificate ...\n";
	print "What kind of certificate are you going to create: \n";
	for (my $i = 0; $i <= $#profno; $i++) {
		printf "  %2d - %s\n", $i+1, $profno[$i];
	}
	chomp(my $answer_ext=<STDIN>);
	# convert input back to array indices
	$answer_ext = int($answer_ext) - 1;
	if ($answer_ext < 0 or $answer_ext > $#profno) {
		print "ERROR: invalid choice. No certificate created.\n";
		exit 2;
	}
	my $name = $profno[$answer_ext];
	my $cert_ext = "crt_". $name . "_ext";
	my $cmd = "";
	
	foreach my $var (@{$profiles{$name}}) {
		print "Enter the value for ", $var, ": \n";
		chomp(my $answer_var=<STDIN>);
		$cmd .= "export ". $var ."=". $answer_var ."; ";
	}

	# determine the hash algorithm to use in the CSR, CLI has priority over config file 
	my $md = $ca_conf{$ca}{'default_md'};
	if (exists $opt{'md'}) {
		$md = $opt{'md'};
	}

	# create sub CA CSR
	my $cmd_csr = "export REQ_DN=req_ca_dn; export REQ_EXT=req_ca_ext; ". 
	     "umask 0077; " . $openssl ." req -config ". $file_conf_req ." ". 
	     "-verbose ". 
	     "-new ".
	     "-newkey ". $dsig ." ";
	foreach my $o (keys %{$alg{$dsig}}) {
		$cmd_csr .= "-pkeyopt ". $o .":". $alg{$dsig}{$o} . " ";
	}
	$cmd_csr .= "-" . $md ." ".
	     "-keyout ". $ca_conf{$ca}{"dir"} . $ca_conf{$ca}{"private_key"} ." ".
	     "-out ". $ca_conf{$ca}{"dir"} ."requests/". $ca ."_ca_req.pem "; 
	print $cmd_csr ."\n" if ($verbose > 0);
	system ($cmd_csr);
	my $r = $? >> 8;

	# determine certificate lifetime, CLI has priority
	my $certlife = "";
	# Note that it is not required to set both start and end date
	if (exists $opt{'start'}) {
		$certlife = "-startdate ". $opt{'start'} ." ";
	}
	if (exists $opt{'end'}) {
		$certlife .= "-enddate ". $opt{'end'} ." ";
	} elsif (exists $opt{'days'}) {
		$certlife .= "-days ". $opt{'days'} ." ";
	} else {
		$certlife .= "-days ". $ca_conf{$root_ca}{"default_days"} ." ";
	}
	# create sub CA certificate
	$cmd .= $openssl ." ca -config ". $file_conf_ca ." ".
	     "-verbose ". 
	     "-name ". $root_ca ." ".
	     "-in ". $ca_conf{$ca}{"dir"} ."requests/". $ca ."_ca_req.pem ". 
	     "-extensions " . $cert_ext ." ".
	     "-extfile ". $file_conf_ext_ca ." ". 
	     $certlife ." ".
	     "-md " . $md ." ".
	     "-out ". $ca_conf{$ca}{"dir"} . $ca_conf{$ca}{"certificate"};
	print $cmd ."\n" if ($verbose > 0);
	system ($cmd);
	$r = $? >> 8 unless ($r != 0);
	print "...done.\n";
	return $r;
}


sub cmd_req() {
	my $conf = $pkidir . $file_conf_req;
	my $profileref = parse_req_config($conf);
	my %profiles = %{$profileref};
	my @profno = sort keys %profiles;
	print "Create certificate signing request (CSR)...\n";
	print "What kind of certificate do you want to request? \n";
	for (my $i = 0; $i <= $#profno; $i++) {
	    printf "  %2d - %s\n", $i+1, $profno[$i];
	}
	chomp(my $answer=<STDIN>);
	# convert input back to array indices
	$answer = int($answer) - 1;
	if ($answer < 0 or $answer > $#profno) {
	    print "ERROR: invalid choice. No certificate request created.\n";
	    exit 2;
	}
	my $name = $profno[$answer];
	my $req_dn = "req_". $name . "_dn";
	my $req_ext = "req_". $name . "_ext";
	my $cmd_csr = "";
	my $cmd_key = "";
	$cmd_csr .= "export REQ_DN=". $req_dn ."; export REQ_EXT=". $req_ext ."; "; 
	
	foreach my $var (@{$profiles{$name}}) {
		print "Enter the value for ", $var, ": \n";
		chomp(my $answer_var=<STDIN>);
		$cmd_csr .= "export ". $var ."=". $answer_var ."; ";
	}
	
	my @t = localtime(time);
	$name = sprintf "%s-%04d%02d%02d-%02d%02d%02d", $name, $t[5]+1900, $t[4]+1, $t[3], $t[2], $t[1], $t[0];
	my $key_file = $ca_conf{$ca}{'dir'} ."private/" . $name ."_key.pem";

	$cmd_key = "umask 0077; ". $openssl . " ";
	if ($keyformat == 0) {
		# write private key in old SSLEAY / PKCS#1 format
		# "genrsa" in newer OpenSSL versions requires "-traditional" parameter for old format
		$cmd_key .= "genrsa -traditional".
                            " -out ". $key_file . 
			    " -". $keyenc. 
			    " ". $alg{"RSA"}{rsa_keygen_bits};
	} else {
		# write private key in OpenSSL 1.0 default format (PKCS8)
		$cmd_key .= "genpkey" .
                            " -out ". $key_file . 
			    " -". $keyenc. 
			    " -algorithm ". $dsig;
		foreach my $o (keys %{$alg{$dsig}}) {
			$cmd_key .= " -pkeyopt ". $o .":". $alg{$dsig}{$o};
		}
	}
	# generate private key first and use it for the CSR generation
	print $cmd_key."\n" if ($verbose > 0);
	system ($cmd_key);
	my $r = $? >> 8;
	if ($r != 0) {
		print STDERR "ERROR: openssl returned error code ", $r, "\n";
		return $r;
	} else {
		print "The private key has been saved to file ". $key_file ."\n";
	}

	# determine the hash algorithm to use in the CSR, CLI has priority over config file 
	my $md = $ca_conf{$ca}{'default_md'};
	if (exists $opt{'md'}) {
		$md = $opt{'md'};
	}
	$cmd_csr .= "umask 0077; " . $openssl ." req -config ". $file_conf_req ." ".
	        "-verbose ". 
	        "-new ". 
	        "-key ". $key_file . " ".
	        "-keyform PEM ".
	        "-" . $md ." ".
	        "-out ". $ca_conf{$ca}{'dir'} .'requests/'. $name ."_req.pem ";
	# generate the CSR
	print $cmd_csr."\n" if ($verbose > 0);
	system ($cmd_csr);
	$r = $? >> 8;
	if ($r != 0) {
		print STDERR "ERROR: openssl returned error code ", $r, "\n";
	} else {
		print "The CSR has been saved to file ", $ca_conf{$ca}{'dir'}, "requests/". $name ."_req.pem\n";
	}
	print "...done.\n";
	return $r;
}


sub cmd_sign($) {
	my $infile = $_[0];
	if (not -r $infile) {
	    print "ERROR: failed to read input file\n";
	    main::HELP_MESSAGE();
	    return 2;
	}
	print "Create certificate...\n";
	print "Which CA should be used to sign the certificate? \n";
	for (my $i = 0; $i <= $#ca_avail; $i++) {
		printf "  %2d - %s\n", $i+1, $ca_avail[$i];
	}
	chomp(my $answer_ca=<STDIN>);
	# convert input back to array indices
	$answer_ca = int($answer_ca) - 1;
	if ($answer_ca < 0 or $answer_ca > $#ca_avail) {
		print "ERROR: invalid choice. No certificate created.\n";
		exit 2;
	} else {
		$ca = $ca_avail[$answer_ca];
	}
	# cut '_ca' from CA config section name to get actual name
	my $ca_name = substr($ca, 0, -3); 
	my $ca_dir = $ca_conf{$ca}{'dir'};
        if (not -r $ca_dir . $ca_conf{$ca}{'private_key'}) {
		print "ERROR: ", $ca_name, " CA private key not found.\n".
		print "No certificate issued. Please create the ", $ca_name, " CA keys and certificate first.\n";
		return 1;
	}
        my $cert_dir = $ca_dir . $ca_conf{"$ca"}{"certs"};
	# config file for all CAs
	my $conf = $pkidir . $file_conf_ext;
	# if a configuration file for a specific CA exists, it takes preference
	if (-f $pkidir . $file_conf_ext . '_' . $ca_name) {
		$conf = $pkidir . $file_conf_ext . '_' . $ca_name;
	}
	my $profileref = parse_cert_config($conf);
	my %profiles = %{$profileref};
	my @profno = sort keys %profiles;
	print "What kind of certificate do you want to create?\n";
	for (my $i = 0; $i <= $#profno; $i++) {
		printf "  %2d - %s\n", $i+1, $profno[$i];
	}
	chomp(my $answer=<STDIN>);
	# convert input back to array indices
	$answer = int($answer) - 1;
	if ($answer < 0 or $answer > $#profno) {
		print "ERROR: invalid choice. No certificate created.\n";
		exit 2;
	}
	my $name = $profno[$answer];
	my $cert_ext = "crt_". $name . "_ext";
	my $cmd = "";
	
	foreach my $var (@{$profiles{$name}}) {
		print "Enter the value for ", $var, ": \n";
		chomp(my $answer_var=<STDIN>);
		$cmd .= "export ". $var ."=". $answer_var ."; ";
	}

	# determine certificate lifetime, CLI has priority
	my $certlife = "";
	# Note that it is not required to set both start and end date
	if (exists $opt{'start'}) {
		$certlife = "-startdate ". $opt{'start'} ." ";
	}
	if (exists $opt{'end'}) {
		$certlife .= "-enddate ". $opt{'end'} ." ";
	} elsif (exists $opt{'days'}) {
		$certlife .= "-days ". $opt{'days'} ." ";
	} else {
		$certlife .= "-days ". $ca_conf{$ca}{"default_days"} ." ";
	}

	my @t = localtime(time);
	$name = sprintf "%s-%04d%02d%02d-%02d%02d%02d", $name, $t[5]+1900, $t[4]+1, $t[3], $t[2], $t[1], $t[0];
	# determine the hash algorithm to use in the CSR, CLI has priority over config file 
	my $md = $ca_conf{$ca}{'default_md'};
	if (exists $opt{'md'}) {
		$md = $opt{'md'};
	}
	$cmd .=  $openssl ." ca -config ". $file_conf_ca .
		" -name ". $ca ." ".
		"-verbose ". 
		"-in ". $infile ." ". 
		"-extfile ". $conf ." ".
		"-extensions ". $cert_ext ." ".
		$certlife ." ".
		"-md " . $md ." ".
		"-out ". $cert_dir . $name ."_cert.txt ";
	print $cmd."\n" if ($verbose > 0);
	system ($cmd);
	my $r = $? >> 8;
	if ($r != 0) {
		print STDERR "ERROR: openssl returned error code ", $r, "\n";
		return $r;
	}
	# convert certificate into different formats and naming conventions
	my $cmd2= $openssl ." x509  ".
		  "-in ". $cert_dir . $name ."_cert.txt ".
		  "-outform PEM -out ". $cert_dir . $name ."_cert_PEM.crt ";
	print $cmd2."\n" if ($verbose > 0);
	system ($cmd2);
	$r = $? >> 8;
	if ($r != 0) {
		print STDERR "ERROR: openssl returned error code ", $r, "\n";
		return $r;
	}
	my $cmd3= $openssl ." x509  ".
		  "-in ". $cert_dir . $name ."_cert.txt ".
		   "-outform DER -out ". $cert_dir . $name ."_cert_DER.cer ";
	print $cmd3."\n" if ($verbose > 0);
	system ($cmd3);
	$r = $? >> 8;
	if ($r != 0) {
		print STDERR "ERROR: openssl returned error code ", $r, "\n";
	}
	print "The certificate has been saved to these files: \n".
	      " PEM format: ". $cert_dir . $name ."_cert_PEM.crt\n".
	      " DER format: ". $cert_dir . $name ."_cert_DER.cer\n".
	      "   readable: ". $cert_dir . $name ."_cert.txt\n";
	print "...done.\n";
	return $r;
}


sub cmd_revoke($) {
	my $certfile = $_[0];
	if (not -r $certfile) {
	    print "ERROR: failed to read input file\n";
	    main::HELP_MESSAGE();
	    return 2;
	}
	print "Revoking certificate...\n";
	print "Which CA should be used to revoke the certificate?\n";
	for (my $i = 0; $i <= $#ca_avail; $i++) {
		printf "  %2d - %s\n", $i+1, $ca_avail[$i];
	}
	chomp(my $answer_ca=<STDIN>);
	# convert input  back to aray indices
	$answer_ca = int($answer_ca) - 1;
	if ($answer_ca < 0 or $answer_ca > $#ca_avail) {
		print "ERROR: invalid choice. No certificate revoked.\n";
		exit 2;
	} else {
		$ca = $ca_avail[$answer_ca];
	}
	print "What is the revocation reason: \n";
	print "NONE means that no reason is included in the CRL, thus avoiding";
	print "use of the 'unspecified' reason that is discouraged by RFC 5280.\n";
	for (my $i = 0; $i <= $#revoke_reasons; $i++) {
		printf "  %2d - %s\n", $i+1, $revoke_reasons[$i];
	}
	chomp(my $answer=<STDIN>);
	# convert input back to array indices
	$answer = int($answer) - 1;
	if ($answer < 0 or $answer > $#revoke_reasons) {
		print "ERROR: invalid choice. No certificate revoked.\n";
		exit 2;
	}
	my $rev_reason = $revoke_reasons[$answer];

	my $cmd = $openssl ." ca -config ". $file_conf_ca .
		" -name ". $ca ." ".
		"-verbose ". 
		"-revoke '". $certfile."'";

	if ($rev_reason ne 'NONE') {
		$cmd .= " -crl_reason ". $rev_reason;
	}
	if ($rev_reason eq 'keyCompromise') {
		print "Enter the invalidity date and time in the format YYYYMMDDHHMMSSZ:\n";
		print "This is optional but backdating the revocation is good practice ";
		print "if the key compromise time is known\n.";
	        print "Press ENTER to skip specifying an invalidity time.\n";
		chomp(my $answer_kct=<STDIN>);
		$cmd .= " -crl_compromise ". $answer_kct;
	}
	elsif ($rev_reason eq 'cACompromise') {
		print "Enter the invalidity date and time in the format YYYYMMDDHHMMSSZ:\n";
		print "This is optional but backdating the revocation is good practice ";
		print "if the key compromise time is known\n.";
	        print "Press ENTER to skip specifying an invalidity time.\n";
		chomp(my $answer_cct=<STDIN>);
		$cmd .= " -crl_CA_compromise ". $answer_cct;
	}

	print $cmd."\n" if ($verbose > 0);
	system ($cmd);
	my $r = $? >> 8;
	if ($r != 0) {
		print STDERR "ERROR: openssl returned error code ", $r, "\n";
		return $r;
	} else {
		print "The certificate has been revoked in the CA database.\n";
	}
	print "Do you want to update the CRL now (y/n)? ";
	chomp(my $answer_crl=<STDIN>);
	if ($answer_crl eq 'y' or $answer_crl eq 'Y') {
		$r = cmd_gencrl();
	} else {
		print "...done.\n";
	}
	return $r;
}


sub cmd_selfsign($$) {
	my $csrfile = $_[0];
	my $keyfile = $_[1];
	if (not -r $csrfile) {
	    print "ERROR: failed to read certificate request file\n";
	    main::HELP_MESSAGE();
	    return 2;
	}
	if (not -r $keyfile) {
	    print "ERROR: failed to read private key file\n";
	    main::HELP_MESSAGE();
	    return 2;
	}
	my $conf = $pkidir . $file_conf_ext;
	my $profileref = parse_cert_config($conf);
	my %profiles = %{$profileref};
	my @profno = sort keys %profiles;
	print "Create certificate...\n";
	print "What kind of certificate do you want to create?\n";
	for (my $i = 0; $i <= $#profno; $i++) {
		printf "  %2d - %s\n", $i+1, $profno[$i];
	}
	chomp(my $answer=<STDIN>);
	# convert input back to array indices
	$answer = int($answer) - 1;
	if ($answer < 0 or $answer > $#profno) {
		print "ERROR: invalid choice. No certificate created.\n";
		exit 2;
	}
	my $name = $profno[$answer];
	my $cert_ext = "crt_". $name . "_ext";
	my $cmd = "";
	
	foreach my $var (@{$profiles{$name}}) {
		print "Enter the value for ", $var, ": \n";
		chomp(my $answer_var=<STDIN>);
		$cmd .= "export ". $var ."=". $answer_var ."; ";
	}

	# determine certificate lifetime, CLI has priority
	my $certlife = "";
	# Note that it is not required to set both start and end date
	if (exists $opt{'start'}) {
		$certlife = "-startdate ". $opt{'start'} ." ";
	}
	if (exists $opt{'end'}) {
		$certlife .= "-enddate ". $opt{'end'} ." ";
	} elsif (exists $opt{'days'}) {
		$certlife .= "-days ". $opt{'days'} ." ";
	} else {
		$certlife .= "-days ". $ca_conf{$ca}{"default_days"} ." ";
	}
	my @t = localtime(time);
	$name = sprintf "%s-%04d%02d%02d-%02d%02d%02d", $name, $t[5]+1900, $t[4]+1, $t[3], $t[2], $t[1], $t[0];
	# put certificate into default CA directory
        my $cert_dir = $ca_conf{$ca}{'dir'} . $ca_conf{$ca}{'certs'};
	# determine the hash algorithm to use in the CSR, CLI has priority over config file 
	my $md = $ca_conf{$ca}{'default_md'};
	if (exists $opt{'md'}) {
		$md = $opt{'md'};
	}
	$cmd .=  $openssl ." ca -config ". $file_conf_ca .
		" -selfsign ".
		"-verbose ". 
		"-keyfile ". $keyfile ." ". 
		"-in ". $csrfile ." ". 
		"-extfile ". $conf ." ".
		"-extensions ". $cert_ext ." ".
		$certlife ." ".
		"-md " . $md ." ".
		"-out ". $cert_dir . $name ."_cert.txt ";
	print $cmd."\n" if ($verbose > 0);
	system ($cmd);
	my $r = $? >> 8;
	if ($r != 0) {
		print STDERR "ERROR: openssl returned error code ", $r, "\n";
		return $r;
	}
	# convert certificate into different formats and naming conventions
	my $cmd2= $openssl ." x509  ".
		  "-in ". $cert_dir . $name ."_cert.txt ".
		  "-outform PEM -out ". $cert_dir . $name ."_cert_PEM.crt ";
	print $cmd2."\n" if ($verbose > 0);
	system ($cmd2);
	$r = $? >> 8;
	if ($r != 0) {
		print STDERR "ERROR: openssl returned error code ", $r, "\n";
		return $r;
	}
	my $cmd3= $openssl ." x509  ".
		  "-in ". $cert_dir . $name ."_cert.txt ".
		   "-outform DER -out ". $cert_dir . $name ."_cert_DER.cer ";
	print $cmd3."\n" if ($verbose > 0);
	system ($cmd3);
	$r = $? >> 8;
	if ($r != 0) {
		print STDERR "ERROR: openssl returned error code ", $r, "\n";
	}
	print "The certificate has been saved to these files: \n".
	      " PEM format: ". $cert_dir . $name ."_cert_PEM.crt\n".
	      " DER format: ". $cert_dir . $name ."_cert_DER.cer\n".
	      "   readable: ". $cert_dir . $name ."_cert.txt\n";
	print "...done.\n";
	return $r;
}


sub cmd_createp12(@) {
	print "Creating PKCS#12 container file...\n";
	my $outfile = fileparse($_[0]);
	shift;
	my @infiles = @_;
	# add .p12 filename extension if not existant
	$outfile .= ".p12" unless ($outfile =~ /\.p12$/);
	if (-e $outfile) {
	    print "ERROR: won't overwrite existing output file\n";
	    main::HELP_MESSAGE();
	    return 2;
	}
	my ($fh_tp12, $temp_p12) = tempfile(UNLINK => 1);
	foreach my $f (@infiles) {
		if (not -r $f) {
		    print "ERROR: failed to read input file ",$f,"\n";
		    main::HELP_MESSAGE();
		    return 2;
		}
		open(INF, "<", $f) or die;
		while (my $line = <INF>) {
			print $fh_tp12 $line;
		}
		close(INF);
	}
	close($fh_tp12);
	print "Note: The first password asked is for the private key file.\n";
	my $cmd = $openssl .' pkcs12 -export'.
		' -out '. $pkidir .'export/'. $outfile .
		' -in '. $temp_p12;
	# Could ask for and insert p12 content name here, but who cares for it anyway?
	#	' -name '. $p12_name;
	print $cmd."\n" if ($verbose > 0);
	system ($cmd);
	my $r = $? >> 8;
	if ($r != 0) {
		print STDERR "ERROR: openssl returned error code ", $r, "\n";
	}
	print "Exported inputs to PKCS#12 file ". $pkidir ."export/". $outfile ."\n";
	print "...done.\n";
	return $r;
}


sub cmd_gencrl() {
	print "For which CA should a Certificate Revocation List (CRL) be generated: \n";
	for (my $i = 0; $i <= $#ca_avail; $i++) {
	    printf "  %2d - %s\n", $i+1, $ca_avail[$i];
	}
	chomp(my $answer_ca=<STDIN>);
	# convert input back to array indices
	$answer_ca = int($answer_ca) - 1;
	if ($answer_ca < 0 or $answer_ca > $#ca_avail) {
	    print "ERROR: invalid choice. No CRL created.\n";
	    return 2;
	} else {
		$ca = $ca_avail[$answer_ca];
	}
	print "Generating new Certificate Revocation List...\n";
	my $cmd = $openssl .' ca -config '. $file_conf_ca .
		  ' -name '. $ca ." ". 
		  '-gencrl -out '. $ca_conf{$ca}{'dir'} . $ca_conf{$ca}{'crl'};
	print $cmd."\n" if ($verbose > 0);
	system ($cmd);
	my $r = $? >> 8;

	return $r unless ($r == 0);
	my $crl2 = $ca_conf{$ca}{'crl'};
	substr($crl2, -3, 3) = "der";
	my $cmd2 = $openssl .' crl '.
		  '-inform PEM -in '. $ca_conf{$ca}{'dir'} . $ca_conf{$ca}{'crl'} .
		  ' -outform DER -out '. $ca_conf{$ca}{'dir'} . $crl2;
	print $cmd2."\n" if ($verbose > 0);
	system ($cmd2);
	my $r = $? >> 8;

	print "...done.\n";
	return $r;
}


sub cmd_exportca() {
	print "Export CA certificate chain for which sub CA?\n";
	my $ri = 0;
	for (my $i = 0; $i <= $#ca_avail; $i++) {
		if ($ca_avail[$i] ne $root_ca) {
			printf "  %2d - %s\n", $i+1, $ca_avail[$i];
		} else {
			$ri = $i;
		}
	}
	chomp(my $answer_ca=<STDIN>);
	# convert input back to array indices
	$answer_ca = int($answer_ca) - 1;
	if (($answer_ca < 0) or ($answer_ca > $#ca_avail) or ($answer_ca == $ri)) {
		print "ERROR: invalid choice.\n";
		return 2;
	} else {
		$ca = $ca_avail[$answer_ca];
	}
	print "Creating a PKCS#7 file of the CA certificate chain...\n";
	my ($fh_tp7, $temp_p7c) = tempfile(UNLINK => 1);
	foreach my $f ($pkidir . $root_ca_cert, $ca_conf{$ca}{"dir"} . $ca_conf{$ca}{"certificate"}) {
		open(CF, "<", $f) or do {
			print "WARNING: failed to read $f\n".
			      "         Certificate chain may be incomplete\n";
			next;
		};
		while (my $line = <CF>) {
			print $fh_tp7 $line;
		}
		close(CF);
	}
	close($fh_tp7);
	# convert cert bundle to pkcs7
	my $cmd = $openssl ." crl2pkcs7 -nocrl -outform DER ".
	     "-out ". $pkidir ."export/". $ca ."_". $file_ca_certs_export_pkcs7 ." ".
	     "-certfile ". $temp_p7c; 
	print $cmd."\n" if ($verbose > 0);
	system $cmd;
	my $r = $? >> 8;
	print "CA certificates have been exported to file ". $pkidir ."export/". 
	      $ca ."_". $file_ca_certs_export_pkcs7 ."\n";
	print "...done.\n";
	return $r;
}


sub cmd_cleanup() {
	print "Please confirm deletion of the complete PKI including sub CAs, certs, keys, databases (y/n): ";
	chomp(my $answer=<STDIN>);
	if ($answer ne "y") {
		print "\nNo cleanup done.\n";
		exit 0;
	}
	print "\nDeleting the CA...\n";
	my $cmd = "rm -rf " . $pkidir . "export ";

	foreach my $c (@ca_avail) {
		my $cad = $ca_conf{$c}{'dir'};
		# avoid deleting the whole main PKI directory
		if ($cad ne $pkidir) {
			$cmd .= $cad . " ";
		} else {
			$cmd .= $cad . "requests ";
			$cmd .= $cad . "private ";
			$cmd .= $cad . $ca_conf{$c}{'certs'} . " ";
			$cmd .= $cad . $ca_conf{$c}{'crl_dir'} . " ";
			$cmd .= $cad . $ca_conf{$c}{'new_certs_dir'} . " ";
		}
	}

	print $cmd."\n" if ($verbose > 0);
	system($cmd);
	my $r = $? >> 8;
	# restore default ca config from backup
	move($pkidir . $file_conf_ca . ".bak", $pkidir . $file_conf_ca) or 
		print "WARNING: failed to restore config file ", $file_conf_ca, " from backup: ", $!, "\n";
	move($pkidir . $file_conf_alg . ".bak", $pkidir . $file_conf_alg) or 
		print "WARNING: failed to restore config file ", $file_conf_alg, " from backup: ", $!, "\n";
	# keep *.cnf and *.bak files
	$cmd="rm -f root* index* serial* *.pem *.p7c arl* crl*";
	print $cmd."\n" if ($verbose > 0);
	system($cmd);
	$r = $? >> 8 unless ($r != 0);
	return $r;
}


# read required ca settings from configuration file
sub parse_ca_config($) {
	open CONFFILE, "<", $_[0] or die "ERROR: failed to read openssl configuration file ". $_[0] ."\n";
	my %ca_conf_parsed = ();
	my $sel_ca_parsed;
	my @ca_sections= ();
	my @req_sections= ();
	my @req_env = ();
	my $section = "";
	while (<CONFFILE>) {
		print "DEBUG: read ca config line: ",$_ if ($verbose > 5);
		if (/^#/ or /^\s*$/) {
			# ignore comments and empty lines
			next;
		} elsif (/^\s*\[\s*([[:alnum:]_-]+)\s*\]\s*$/) {
			$section = $1;
			print "DEBUG: found config section ",$section,"\n" if ($verbose > 4);
	    		%{$ca_conf_parsed{$section}} = ();
			if ($section =~ /_ca$/) {
				print "DEBUG: found CA config section ",$section,"\n" if ($verbose > 4);
				push @ca_sections, $section;
			}
			# code to find environment variables in sub CA req sections (not used so far)
			elsif ($section =~ /req_[[:alnum:]_-]+_(dn|ext)/) {
				print "DEBUG: found CSR config section ",$section,"\n" if ($verbose > 4);
				push @req_sections, $section;
			}
			next;
		} elsif (/^\s*(\S+)\s*=\s*([^#]+)/) {
			my ($key, $val) = ($1, $2);
			$val =~ s/\s*$//;
			# special case for main directory of each CA: 
			# replace relative path (like .) with absolute path
			# This is done to enable different paths per CA (but not recommended)
			if ($key eq "dir") {
				$val = abs_path($val);
			}
			# add directory separator to all values that contain a directory name
			if (($key eq "dir") or ($key eq "certs") or 
			    ($key eq "crl_dir") or ($key eq "new_cert_dir")) {
				$val .= '/' unless (substr($val, -1, 1) eq '/');
			}
			$ca_conf_parsed{$section}{$key} = $val;
		} else {
			print "DEBUG: found unparsable config entry: ",$_,"\n" if ($verbose > 4);
		}
	}
	close CONFFILE;

	foreach my $s (@req_sections) {
	    	print "DEBUG: parsing CA req config section ",$s,"\n" if ($verbose > 4);
		foreach my $k (keys %{$ca_conf_parsed{$s}}) {
			# find environment variables like ${ENV::REQ_DN} or $ENV::UPN in the CSR sections
			if ($ca_conf_parsed{$s}{$k} =~ /\$\{?ENV::([A-Z0-9_]+)\}?/) {
				print "DEBUG:   found required environment variable ",$1,"\n" if ($verbose > 4);
				push @req_env, $1;
			}
		}
	}
	$sel_ca_parsed = $ca_conf_parsed{"ca"}{"default_ca"};

	return ($sel_ca_parsed, \@ca_sections, \%ca_conf_parsed, \@req_env);
}


# read algorithm settings from configuration file
sub parse_alg_config($) {
	open CONFFILE, "<", $_[0] or die "ERROR: failed to read algorithm configuration file ". $_[0] ."\n";
	my %alg_parsed = ();
	my $section = "";
	while (<CONFFILE>) {
		print "DEBUG: read alg config line: ",$_ if ($verbose > 5);
		if (/^#/ or /^\s*$/) {
			# ignore comments and empty lines
			next;
		} elsif (/^\s*\[\s*([[:alnum:]_-]+)\s*\]\s*$/) {
			$section = $1;
			print "DEBUG: found config section ",$section,"\n" if ($verbose > 4);
			next;
		} elsif (/^\s*(\S+)\s*=\s*([^#]+)/) {
			my ($key, $val) = ($1, $2);
			$val =~ s/\s*$//;
			$alg_parsed{$section}{$key} = $val;
		} else {
			print "DEBUG: found unparsable config entry: ",$_,"\n" if ($verbose > 4);
		}
	}
	close CONFFILE;
	# ensure that default algorithm is set
	if (not defined $alg_parsed{"default"}{"publickey"} or $alg_parsed{"default"}{"publickey"} eq "") {
		print "WARNING: Missing default public key value in algorithm config file\n";
		print "WARNING: Using default value '",$dsig_avail[0],"' for public key algorithm'\n";
		$alg_parsed{"default"}{"publickey"} = $dsig_avail[0];
	} else {
		my $found = 0;
		foreach (@dsig_avail) {
			if ($_ eq $alg_parsed{"default"}{"publickey"}) {
				$found = 1;
			}
		}
		if ($found == 0) {
			print "WARNING: Unknown value for public key algorithm in config file\n";
			print "WARNING: Using default value '",$dsig_avail[0],"' for public key algorithm'\n";
			$alg_parsed{"default"}{"publickey"} = $dsig_avail[0];
	       }	       
	}

	return (\%alg_parsed);
}


# find all environment variables that need to be set for each config section
sub parse_cert_config($) {
	open CONFFILE, "<", $_[0] or die "ERROR: failed to read openssl configuration file ". $_[0] ."\n";
	my %parsed = ();
	my %sections= ();
	my $section = "";
	while (<CONFFILE>) {
		if (/^\s*\[\s*crt_([[:alnum:]_]+)_(dn|ext)\s*\]/) {
			$section = $1;
			print "DEBUG: found config section ",$section,"_",$2,"\n" if ($verbose > 4);
			next;
		}
		# ignore comments lines
		next if (/^#/);
		# don't parse the initial section as it cannot contain a certificate type definition
		$sections{$section} .= $_ unless ($section eq "");
	}
	close CONFFILE;

	foreach my $s (keys %sections) {
		my $conf = $sections{$s};
	    	print "DEBUG: parsing config section ",$s,"\n" if ($verbose > 4);
	    	@{$parsed{$s}} = ();
		while ($conf =~ /\$\{?ENV::([A-Z0-9_]+)\}?/gs) {
			push @{$parsed{$s}}, $1;
			print "DEBUG:   found required environment variable ",$1,"\n" if ($verbose > 4);
		}
	}
	return \%parsed;
}


# find all environment variables that need to be set for each config section
sub parse_req_config($) {
	open CONFFILE, "<", $_[0] or die "ERROR: failed to read openssl configuration file ". $_[0] ."\n";
	my %parsed = ();
	my %sections= ();
	my $section = "";
	while (<CONFFILE>) {
		if (/^\s*\[\s*req_([[:alnum:]_]+)_(dn|ext)\s*\]/) {
			$section = $1;
			print "DEBUG: found config section ",$section,"_",$2,"\n" if ($verbose > 4);
			next;
		}
		# ignore comments lines
		next if (/^#/);
		# don't parse the initial section as it cannot contain a certificate type definition
		$sections{$section} .= $_ unless ($section eq "");
	}
	close CONFFILE;

	foreach my $s (keys %sections) {
		my $conf = $sections{$s};
	    	print "DEBUG: parsing config section ",$s,"\n" if ($verbose > 4);
	    	@{$parsed{$s}} = ();
		while ($conf =~ /\$\{?ENV::([A-Z0-9_]+)\}?/gs) {
			push @{$parsed{$s}}, $1;
			print "DEBUG:   found required environment variable ",$1,"\n" if ($verbose > 4);
		}
	}
	return \%parsed;
}


# retrieve list of supported hash algorithms from openssl
sub ossl_get_digests() {
	my @found = ( );
	my $hash_get_cmd = $openssl . ' dgst -list';
    	print "DEBUG: retrieving openssl digest list...\n" if ($verbose > 4);
	my $hash_get_out = qx/$hash_get_cmd/;
	if (($? >> 8) == 0) {
		my @hash_get = split(" ", $hash_get_out);
		foreach (@hash_get) {
    			print "DEBUG: parsing output: ",$_,"\n" if ($verbose > 5);
			if ($_ =~ /^-(\S+)/) {
    				print "DEBUG: found digest ",$1,"\n" if ($verbose > 4);
				if ($1 eq "sha256") {
					# insert sha256 as the first element (default value)
					unshift @found, $1;
				} else {
					push @found, $1;
				}
			}
		}
	} else {
		# use a hardcoded fallback list of openssl call fails
		@found = ( "sha256", "sha384", "sha512", "sha3-256", "sha3-512" );
	}
	return \@found;
}


# retrieve list of supported named curves from openssl
sub ossl_get_curves() {
	# always include NIST curves first 
	# These more common names are not used but understood by openssl
	my @found = ( "P-384", "P-256", "P-521" );
	my $curve_get_cmd = $openssl . ' ecparam -list_curves';
    	print "DEBUG: retrieving openssl named curves list...\n" if ($verbose > 4);
	my $curve_get_out = qx/$curve_get_cmd/;
	if (($? >> 8) == 0) {
		my @curve_get = split("\n", $curve_get_out);
		foreach (@curve_get) {
    			print "DEBUG: parsing output: ",$_,"\n" if ($verbose > 5);
			if ($_ =~ /^\s*([[:alnum:]_-]+)\s*:\s/) {
    				print "DEBUG: found curve ",$1,"\n" if ($verbose > 4);
				push @found, $1;
			}
		}
	}
	# fallback is already defined above, no additional processing needed
	return \@found;
}


sub main::HELP_MESSAGE {
        print STDERR "Usage: $::PROG [options] <command>\n".
        "  Commands (select one):\n".
        "    init                    Create files and directories needed for an OpenSSL based CA\n".
        "    edit                    Change CA configuration settings\n".
        "    newca                   Create key pair and certificate for the root CA\n".
        "    newsubca                Create key pair and certificate for a sub CA\n".
        "    req                     Create a certificate signing request for an end-entity\n".
        "                            The type of certificate can be chosen from all definitions\n".
        "                            found in the OpenSSL configuration files\n".
        "    sign <file>             Create a new certificate by signing the given certificate\n".
        "                            request <file> by one of the available CAs\n".
        "    selfsign <csr> <key>    Create a self-signed certificate using the certificate\n".
        "                            request file <csr> and the private key file <key>\n".
        "    revoke <file>           Revoke a given certificate <file>\n".
        "    gencrl                  Generate a certificate revocation list (CRL)\n".
        "    p12 <out> <files>       Create a PKCS#12 file from all given <files>.\n".
        "                            The <out> parameter should not contain a path because\n". 
        "                            the file will be placed in the 'export' subdirectory.\n".
        "                            The input files need to be in PEM format and contain at\n".
        "                            least one private key and its corresponding certificate.\n".
        "                            Additional certificate files can be included as needed.\n".
        "    exportca                Create a PKCS#7 (.p7c) file that contains all CA certificates\n".
        "                            This may be used to publish the CA certificate chain\n".
        "    cleanup                 Completely delete the CA in this directory including all\n".
        "                            private keys, public keys, certificates, files, directories\n".
        "".
        "  Options (must precede the command):\n".
        "    --start=<YYMMDDHHMMSSZ> Certificate will be issued with specific startdate.\n".
        "                            Year can be specified with 4 or 2 digits.\n".
        "                            Not supported for root certificates (newca command)\n".
        "    --end=<YYMMDDHHMMSSZ>   Certificate will be issued with specific enddate.\n".
        "                            Year can be specified with 4 or 2 digits.\n".
        "                            Not supported for root certificates (newca command).\n".
        "                            Use --days instead.\n".
        "    --days=<NUMBER>         Certificate will be valid for <NUMBER> days.\n".
        "                            Use of --end and --days is mutually exclusive.\n".
        "                            WARNING: If used with --start, the --days <NUMBER> is\n".
        "                            calculated from the current date and not from --start.\n".
        "                            Default value for root certificates: ".$lifetime_root_ca."\n".
        "    --hash=<STRING>         Use the hash algorithm <STRING>\n".
        "    --size=<NUMBER>         Use <NUMBER> bits key size when creating RSA key pairs\n".
        "    --curve=<STRING>        Use named curve <STRING> when creating EC key pairs\n".
        "    --keyenc=<STRING>       Symmetric algorithm to encrypt private key files\n".
        "                            Default value: aes-128-cbc\n".
        "    --oldkey                Create RSA private keys in traditional SSLEAY format\n".
        "                            (OpenSSL 0.9.x backwards compatiblity)\n".
        "                            Not supported for any other algorithm than RSA\n".
        "    -v   --verbose          Be more verbose (use multiple times for more details)\n".
        "    -d   --debug            Enable debug output (use -d -and -v for maximum details)\n".
        "    -V|--version            Show program version\n".
        "    -?|-h|--help            Show this help\n".
        "\n";
}

